<?php
/*--------------------------------------------------------------
   GambioCustomersSignupsCount.php 2021-08-18
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2021 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
 -------------------------------------------------------------*/

declare(strict_types=1);

namespace Gambio\Admin\Modules\Statistics\App\Overview\Model\Entities\WidgetDefinition;

use Doctrine\DBAL\Connection;
use Gambio\Admin\Modules\Statistics\App\Data\Factory as DataFactory;
use Gambio\Admin\Modules\Statistics\App\Overview\Factory as OverviewFactory;
use Gambio\Admin\Modules\Statistics\App\Overview\Factory\Option\Predefined\TimespanDropdown;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\Collections\Options;
use Gambio\Admin\Modules\Statistics\Model\ValueObjects\Data;
use NumberFormatter;

class GambioCustomersSignupsCount extends GambioWidgetDefinition
{
    private const TYPE = 'gambio.customers.signups.count';
    
    private const VERSION = '0.1.0';
    
    private const ICON = "data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAQQAAACoCAYAAAAGoZArAAAACXBIWXMAAAgAAAAIAAHXmQl7AAAAGXRFWHRTb2Z0d2FyZQB3d3cuaW5rc2NhcGUub3Jnm+48GgAAIABJREFUeJztnXucXFWV739r73OqutN50EBeGAhBHsqIOjrXFyKiXhVfo6O0CgESBHpQUT/e8a2T5o7zUO917qAgTcAOEB525jqDoKCXUePz6vgaHANXEAiBBExISDrdXVVnr/W7f5xTlQ50kuqkOp2k9vfzqc+n6tQ5Z++q7v2rtddeey0gEolEIpFIJBKJRCKRSCQSiUQikUgkEolEIpFIJBKJRCKRSCQSiUQikUgkEolEIpFIJBKJRCKRSCQSiUQikUgkEolEIpFIJBKJRCKRSCQSiUQikUgkEolEIpFIJBKJRCKRSCQSiUQikUgkEolEIpFIJBKJRCKRSCQSiUQikUgkEolEIpFIJBKJRCKRSCSyP5Cp7sDBzOWXX17207qPLqc2wySdSdLt7nxnxgDAe0+EAOc8AzJ4egKAedKbYwbAexJZfm4WMnjn6b0xC8X1CDDviQwwbwQAb55ABrP8ft4bswzwPqFPjFkN8Ilnhqw4Nz/HLCFQQ5IkrNUAn+T3M00IVKFJwsRSAlWklrBanIsqoIkSAFJLWUEVaXE8JCnTkhGjgJaKc0LKUYyiVCqxpCWOFM9tO1VkJFPVSm9v78hk/b0ieyYKwgToHxycJSPV10HkNQK+mMQiAL6JS42ACWAgDSL5a4GCMIEYhQbCgPxBgQlhAA0QA4tjxfv5a5qIqFAMoFFgKJ5LcQ+wuB4wkiYiRRs0EibOab1fBAwiBqOJwAxiImYwMQiMoIGS90GK8ynmHBSA2Zj3pHjPYHmblt/LIAaDieRtKMTEzJz3KgILQWuJ81vg7HEdTR7r7T37CQCctD9qZCeiIDTBVTfc8DynuFQEbyXR2ex1ApBAMSCLQYodgx7IB5kwH3AAjcXgF4GKwMinXp8/hxSi0BCafCAChdjkg18hYiQNQCEGZmQhCiImRjMZI1aQXEhAc3AKYIxY7RjUuQjlg92suBfMQCmEJW/TwauJmYwRgrw9MVghFiJmZoWY5G2oiQnEhLbdmXvomc9c8NAZZ5wRJuUPHGkQBWE3LL/++kU0+TsAb8bEvyuiMVDrA5cmEMVOFoGYgGOEYsygfpp1wOJ8UTZEYcz1IsWgRkMUIDusAzREASaAErkQSeMayQdyYbUYbGexEhS/9LnY5MKyw4qweh8FJiZmYoXoFEIkRXtm+efyXjFGCGyMQAjE1NREJH+oVRNJH1iy5F1rRSRaDJNEFIRxIClXX3fDe0VcH8hpE77BmIGJMYOs8Ws+ZvCTYiJ1IYABTusWgHBni0Aoudk+xiIgZMfUIh9MijGDn41+5K+NhTkPMbAYpHXxYW4djJ3KQGgcawEUFoEUFoBQ8mlA3QogDA6KMf2yQiBEciEwyacJ+T1hWghBw1pQLaYQYqJmWhcFUXNIt1WG/L3ve1/P9pb9wSMNoiA8hSuuuGJ62jXzKgBv28tb7LO/gIVVcSj6Czyg+bQAZlqIQkMILBfHQjAUhRCYmkAsFKKQSilLJfzhvPPO+2Nr/uqROlEQxjAwMHBYkPRWgn820Wujv6AF/gKBOUCtPm3Q/BothEA0F4hQPHcl9+hFixc/0tJ/gjYnCkLBwMDAYZmk3wb4nOaukHsE+D5g9yixQRxHnTlyD0uK9eeNJcXGEmCWLyOG+vEAb547nwOY2W6XFJ+27FgsI+5qSTFJElZrQJIYUQUsLZYgNaEmKYEK0jRlpQqkxRJj0N0vKWIUCMXxUihxBCP5UmMIHAFQKpeJ4WGEcpkAUA4dHOZQWvK+U5jOEOrhEPFPFQImUAli4vLjQcRKobx+6dKex1r1f9DuREEA0Dc4WDpquPYNCE/b7YkiVZArSH9179Kz791P3Ws7SMrAwMCR5jsXBNoMcWoSCutgzHORYOKcZcPJuve+95wtU93vQ4EoCACuvu6GL4B47+7OocidQWsfet8FF6zbX/2KAF++7rojykyONaeJqGsIQXCuEIZgJLUEPLR06dLKVPf3YKftBeGaFStfbeCt2NV3IRJI+2TvkvOu2L89i9Tp+973kjkPPPLM1Ov0IGISnIkLlqapZpkzcc46pFY599xz18UlyX2jrQXh8ssvL3fO6v4FiePGe1+AGoHzL15y7jf2d98iO0NSrlqxYmHi3EwRsRCciTgTl5nLnNWcs06pbV66dOmTU93Xg5ndxt4f6nTM6v7LXYkB8sCi90YxODAQEV6ydOnaYLaNgKYptQQqWVaWqKUyNIR0+uDgYDOh5JFd0LaCMDAw0AHKh3Z9hlxx0ZJzb95/PYo0AY/o6lo3Omq1LDirSWbO1axa81atVi2UlEND5YkHkkUatK0g1Fz6ToBzxntPBPenzJbt7z5F9kxPT4+WUd1QItRYVgO03EHtYKeamQ7LEx0k23oqvC+0rSA48pxdvUfIp6LH+sClt7d3pJroduczc1VnRqp1UjvZpbO8t1WrVqVT3ceDlbYUhOUrVy6gyEvHf1fuuei8c765f3sUmShHdHQ8aaRaB9VVvPlKxUb8qA0PJ7Z27dooCHtJWwoCTc7ArpKZ0K6JS1cHPj09PbVpmFHpJNW6oGbTtctMZ86kTlu4ME4Z9pI2FQR7+XjHBVD1+Pr+7k9k76gcno6OjibmR0ZsyA/bUJra1q1bLdmyxfr6+tryf3tfacsvTQTPG+cwKfjtJXEH3UHDhl/+sqbTqTpzps4008PMtDp7tlarVZ3qvh2stJ0g5B5oWbTTwXybsArlF1PTq8je0NfXF/z27ZYMDdnmNDVV1QWAnnLKKbps2bI47dsL2k4Qrr12VTfI6WMONbYMm9kfpqpfkb3kqKOyEIIeqapdXV22Zs0aW716tUU/0N6RTHUH9jdaqkwXk53yF0iRf0CcbJzq/kUmxtHe6/1HHw2sW2ddXV22bNmyKAb7QNsJQiLSqXUx4JhsQKCJcb+lAL9icHB6abj6UgNeIiJHATwMwDCITfRyt6n/ySVL3/3QvrbT19fnFhx7wstN2OWMNDrSkSJih3WmP+7p6RmdyP2+fN11R3SgdIxpNgdOyhDxJIYdsc28rcfo6Ibe3t5sX/vdLOvWrbMygO7ubuvp6THEDM37RNsJgpEOEEU9hZhwR+5C7ybdGXXttTecFDw+LCO1cyAyTYociISoAAqBiokmYnr1wA3/Ls5d/eiD993W19dnE21rYODGEzKxD5jYcUKnJnlyVidQ0vSJSuVuAHsUhL6+Pjd30aLnpUhfpKbzFUGdFyWcglChGQEFRdExo9J/7U33Vst29wcWL962V1/SBOju7rY1a9awt7d3wt9P5Om0nSDUqkCa1vMd5mnLRKBCmCMn7ddlcHCwtGW4+kkV+YQApSKFWgCpAHIxQF0YqAQUTp5L2uXzFx5/Tv+KWz7Ru+RdTfk4+vv7p0ln13mZ8U0EIJRA5O0UORGVECuH8h4/7/LlN81lYm8DMd9gCpHMOagZDKA6QglROKgYDDCBw4klxbHXXH/9b95z7rm/m0wT/qyzzopThBbSdk7FJEmY/xLLjoHIPN/hZJkH/f3907aO1G4VkWUAUogEwGUEAnZ6uMwBGYhAIID5MRF5niAMXnXtdbuIrtzB8oGVL5FS11VQvpmgCVwGSEYiAAwkAynBQTJg94mLr/zqjcehjAvFydz8OgYRBpoEQDIHycwxUBiEEkQkAyQTYXAkTf1zrv7qzadO5g7EKAatpe0EAagiHyhQgSgEhTA4nQwLgaRIuetfCLwegAIIIAPAIEAGQUawEATmQuCQCwMQWJwLIHXe/6/+gZuetau2ll9/y7Mp+GsIusVJJpCGCDiHDCIZKQHMB3HW0bHLz7t8+coFqcc5NPNGBgfJnENGSi4MxT0AZF58RjKYMIjLjwE+MzJIYrO3bK9MOGltZGpoO0FIkoSEKFFPNV5PU06jcy0XhGuuW7kEwH8FEAApBktuBQDcKJAvk1hM45nw/h2gfIaGX7I41zlkuVAwCMSJs0/u6hdXajWfi0tuBQDIBMiccKRhGYhkcMhIBgyN3+fBwUHPMt5EggCy+uA3ulwYIA87b9/zwf+rCf9FJXwHwG+FHKVZMLogZPBeMgAZBUdce+3NR7X6u420nrbzIVRrQOKhO9dHKOoE2CRYCJCPCpAVKxnK+jTF4QcuK19w4YU9m59yyU8ArFh+3cp3EvhEPihFJa/NoCCOe7JafXFx3k5YYnSSZiSNpDrBFgFWGt1xAF8poFKgJNRBtNqRjft5twwPnwBJjwAZHKh0XgEq1BTKf7voovPuGeeyhwcGBn6b+fKLhXY4SSWhnlBBoub0GADr9/kLjUwqbWghpGRhETSsA+TORedbKwjLr7vpuQAXFb6CTIBMiABgbba9tHgcMWhw0fmLv0bwKwRzX4KrWxYMDPKK8a4xS5jP86XmhN+tTSt95MIli3/scodibhmgMPcpoaOzc9zPK5Kc6IFMhMHENaYIAH6yCzEAACxdurTC0aGfADIEkUyMwZwL5hhEpNR/220xeckBTtsJAlCFwKkUvgMUXn0RKK3FUwbq8yE7nIcEAgQZBMubKUXW3Vm+XsQNQxBypyACRDLIU0KvC3xiJLAOop+9aMm5V7+vJ2/DYEZBEKLwW7iMwoBtu1gVpBxpwgBxGSSfMsBJZcPDC+7eU597e3uzoHhAjEGSJINI5syCmQUMDUVBOMBpQ0EAyDHBSALN6ylCW20hGGR2bnbv8AMACCb+581c39PTMwriHinm8XWnowAzxzu/DDx84sJnfPyi83b+FSe8OuR+A1KCSN6n2i4sBArLLPwAYgzwyEg+3tfXXPXlTqltNueCGoMzC4BkKElWqiZtN0U92Gi7P5BZQiemrBdPrVc+JsxaXGzcUaZTGFhYIg6iJE1roenMwCLYTCLASb4qQhppHeOde9555w2P2w8PpTHAORXSSChJxdZdpQ0Q54DMjAYPdQaF+aajOGu1I2voHMqcqimoTKCJplqrVWOeggOcthMEoAYg1bweIhq1DyUvU95SC4EChSAjWS/hrnDQkudEQh6KeX9xfR6wNCHpMhWDk+BYBDwBKkLr7KyN+3m9MZjAxIs6QI1ili8nNkV39yi3DlugiCKBuuBMoWqdE/rckSmg7QQhSRKa1qsZF9WSmVdPpvMtFQQRM1BCfUqSF3uFJpY23Q5Jc3kQk5LMqy+z+cEJAPRQT8mMZiiiFQmvtWnjxyHQMRhgnlAr+p6g+cE8PDxMNR+ShEpLlZKHNbvExfDiA5y2EwRUASR5tWWh5HsYinLlTltrIcDEKAgonJZgLgoVNJ+/Nf9FZ0B9IJNWxBg0jZjmYdIiOjbsGLuqhuh95o1mBiOp3kNpmNCve1qSLARnpKmRWipBMVKNgnCA03ZORU1SknkOhLqVkJdhh5pvrYUA5xSQIuIwj0R0cFlaKjXdjoiEsWHHgMskD25qGlEzyZcjG2HHQoaurvGnDMId4ck+kcys7gdpjqGhIar6ALisLJJ5ryELIejMmXHKcIDTfhYCKnmlcexsHYAwx9ZaCEYzyZcc87k/xShU0wkIgpmJk4wQpcHy4KKJThm8iriMNHNE3Q+huwyCQJIJA5E7IxUCTSxpejDPmzePm4aqoURoKFEBp9PYoWGoHC2EA5w2FARgx5Kj5IO0vvOxxT4EQAywQEAdnNIVTsGJThnIQDh1AqVwwua7h1elBQdRCLUI2dau6szxLQTHQC9mpuYBBb3W0ubbXAPgmA6XGaC1atU62Kk1C/pkuRIF4QCn7QQhtZTBmwpdY8mxHrEIbe26o8u3GWdjwo5VAE20eaei1CMFdwo7xoQ6qmImhdlPR/XwSkKBTeOeT2MwmiUQZeLV1IysNS0I87dsYdbZGYzUaTJdK0nFlFS/6YkoCAc4bScIFQAJoBSaUPLQZQcDnZkLLV9lyJ15uUMxT4QC1ZI23Y6ZmeSbhBTMVykm6lT0hJpHRqM5OrUiB0Nlxozx+yHIxEjzXmnUJN8M1rQgnHjiifzPBx4IHdapta6gpNfDSC0tWNByHwLJaar6WpIvFZH5AOYB6ASwnuQG59zvnXO3i8jDrW77UKTtBCFNlDRpRCqO3c/g2dopg+VLmkGAesITBUQxOoFmHJREcPV7iFPBxFI3qFMDJTjkm5SMNO+oMyqVcTvijIFJYgTUqZoRCml+2XH16tU45piTM5sFHRketplmOhyC3v+MZ7TMQsiy7OUi8lFVfQ2ATpGnxzyJCEhCVa9Q1V8BuN45d5WIVFvVj0ONthMEAIBAhXlyVUpuHYjQrMWhy4RTB8sKH4WS+b6JMAELQZwEaO5DoIM6UIut0U2TMFE1BoJKT/WSR07uKv2gOhdUzZK8LVWv1sly04Jw8sknc1OWhZk1asl7VUA3dHcbVq/eZ0EgebKqfg7Am8Yer1RrWLfhMWx8YjNGK1XMnzMb82YfgcMPm1W/7gUAXqCqHwwhfNp7f3NMrvJ02k4QEk2ZSWYc4zsQl6cU89bafxCxfFPR2LBjEFoqT2yVgeIzFGneigClCQlC0GDifGYwc5ZbGqTX0SNnjdsPI4Mza4QdpyzpKJu3ENacdRaP/upXw3AIWq1W9ZRTTtElr3yl7usADCH0qOoAgGkAsH1kBDffege++d0f4oc//xWqtdrTrnn28Yvwxle9Au988+vx7OMXAcAiADeq6ltIXiAi+y2x7sFA2wlCBRV4pCooQpdlx34G863N2CtiZpCdwo7FQTGBf0HSKx1DnruwWDKcgPkOAExSddTgC1+AqZjzplw/fnoCp+OEHdM33+Zll8EtXJgB0Hnz5tkZZ5yxz76DLMv+FsAnAEgWAq655ev4/FUrsPGJXS+eAsA99z+Ie+5/EF+85gac/edn4tOXXowF8+cCwDtVdRHJt4jI4/vav0OFtgtMAgDWcyAAKszn5QDUW2t9CAoxB5fnSCSCSJ4SLZSad16KkyBEsEZOwzwf40T6ISEYyWDG4MwFn+Tp1Ia7u8fth5oF5zQ48/l2a7iso7N5p+KyZcu4du2x4eKLLw49PT37LAaq+hER+SQA2fzkVvz5hR/ER/72i3sUg7GYGVb+yzfxorecjTu+/6P64Rep6jdIjrtZrB1pO0FItUQpshaxXrWpsBCy0NpyAh5OCQYIinyEDISEUhPZjuuYmQHInEgmIg1hmEg/mMcQZEIJ5qwQBgY88si456clydR8ULXgVYNPNNgEpgwiwr6+M0Ir5ughhDNJ/j0A/GHtOpz2jiX4wc9+udf327Z9GO9+/8cwsOrW+qEXqeo1eYm/SNsJAgCMzbgszhXBQlDf4sAkQio7wo6lyKosmZayctP3MCZ5TsM8lBiQTIwT8pJLCCZ0AUl+fT1hSfcuLARjWquHHYci7NhXkv0eQ0CyG8BKAH5oeATvvvRjWPvohn2+b1DFhy77HL79g0YWunNU9ex9vvEhQNsJgpaUaORA4I7gJNCsxaHL4nWLc8iIPFNxPduxqs5r9h7OySyTHZmOhQxS8k3nUwCAlKma5FMGMcunAWmazZ07d9zPmwWt+kRD8BrgXDbNdWWqW/f7/0oI4WMADgeAiz9+Gdbc90DL7q1quOAjy/DIhob74G9INi3UhyptJwgFRdESp8J6OLGob/Gyo5Ab6jUQnEhGKfwA6l7UzPXX3HDD8RApN2ogFIIiYZf7FMeFaUl3pDJjUOeDMwv333//uOf71EazEEIthJBkaailIVhXl/b396cTaXdfIHmUiFwKAD/691/jtrtWt7yNrduG8NkvXV1/ucjMelveyEFG2wlCGlKisA7MzChi9fyK1mKn4iMPPHAvgG151qSitkGe2+C4rwzc8MrdXds/ODiLcK/aqQaCz2sgGGRCP5UuZNZIZZammTMLqhbmzZs37ucN5fIQXFc2TaZntTQL1SwLpWo1dHZ2lupz7RDCe0IIm5t9kFw4kT6b2btQLC9+6gtfmsilE+Lmb9yBe+5/sP5yyaQ1dJDQdoIwmpcyVClyA+Rr8nkqdu9bO2Xo6+sLgP0Q9ZyKlJDHJSBLnLyuf+D6s/v7rz+mr6+v8Xe4YnBw+rXXX/9nMlo7xxQdTpAVOQ2DowtqNlId2vTgbpp9GiRUnQXn3Y4chyLZml2cv3HNmhGMVLJaOQR6nx3mfTYyMhI2dXTYqlWrUpIiImUA3RN4TOh/jeRbgHzZ8Je/3VVP9x1Vw+Dt3663+XySx0xaYwcBbReHUCqVyMwMIko4c1LkRACt1RYCAKQM3whMnkNx0/P9E40VDgXkRNfhnnnUohNq/StWbhfCYbSaKrzCaAIGc/mmKAevZmZwyc8+8IEPTMipWJWalaQjy6N4gzGlJoDO37Jl3M/b19dnfQMD22ZVtaMedqyLFumfdnbq6tWrTUSoumPR4cobvoZK5eldevYJx+HMV758ol8ZSHar6qkA8K3v/mDC10+U2/9tNZZ96C+BPA7sTQCunPRGD1DaThAA5KsMQhODmcCknk5NWl/FfOnSpZWvXHddv4O70AEdINRIc3nyFGWe55AO6DAxA31eHAXQPMEpjfRKUJ2TNRc1WfB1LM57U7VAUEtl0SwLpijp7tbZ+pYurQwODtojj8BvWwA9eft2PeMNbxh36fEfrvwqNj+59WnH3/WW1++VIGRZdrxzLgGAH//iNxO+fqLcc/+D2PzkVhx+2CyY2YmT3uABTNtNGUp5chIFndLlSUTrm4Ymw0IAgEvOP//RVKtX0rDOyOCcy0ucjVk5oDBAfCZFABFdkQIdPhPnRuHCzy5acs5eLcAbqaoanLcQVAOkMyvDZRtOPHG3n7enp6e2bdua6rKzzspaEWDULM65Rtm3Rx/7435pc/0fNwIARKStS861nSCMYBR5YlXLHYuASRGcZC32IYzlggsu2Lh+7f1Xeee+ZbAtRobcP1BffZBCAJCJz6seEa4WAv9Qluz2i88///d723aHdWqpLJmIZEGT4IOGkGo4eePGPX7evr6+/V5uXUTm1p9v3DKhBZW95rE/NnJDtLUgtOeUAVQ4Z2L5fgYTZyJm3pJJ/cfv6+szAD/v6+v7xfxjj10U6I/25uc4SAeAlB41NRvxwk1E8vjwtGTth3t6Rve1XdVtNaYdw5Wqtw6julmJjYyO2po1aw7I3X4kGymlyqXSfmmzs6MRvbzP3/fBTNsJQlk7a0EqjdTrEKhjvgUagnGrKreaQhj+UDwmnQsvvHAzgOYD/6cYkhvq+Q3mHnkE1q1/bNLbnHvkEfWn+x4KeRDTdlOG7aVaFeIUrghdppghnz4I4iaXAwEzawzK445ZMOntlUslPGPeHAC5GE16gwcwbScI80ql4bzOgSgEufe+SD6i4GFT3b+phmQnyeP29ABwZAvb7B77ulQq3YvConntaS9tVTO75BUvfgE6O/KoZRH5yR5OP6RpO0Ho6elRCjbB8vRpzL2LJgYTsbl7vsOhTQjhZar6hz09SF62r21Vq9XnZln2e1XdnGXZvdVq9U+AvBYFgDsA4HWnvwzpJNeIPfOVp9Wfjnrv75rUxg5w2k4QAMDDPwpX1DiEU+fydX8x94zBwcH94keIAEmSfFVETgAAETnJe798zNu3AkD3rJk49y/eNO71reDIw7vx7j8/s/7yLhEZt2Buu9B2TkUAoIYH4OQkgMWOR8lrJgL+ydHRZwLY6yW+Q4l/vOYGPLju0T2eNzI6vmP+F3f/Dh9Y9g8AgP99x13/veOEP9spV9T7P/P3LxAB3vCq0/D6008FgOfX3/Pe/2sI4T4ROeET73sPbrntToyMNl/Polk+dslSzOiaBgAg+fmWN3CQ0ZaC4JHeFxheC3HmxMwEZgaTvIrTcxEFAQBw+3d/gJ/9+rd7ff39D63D/Q+tq79c/NT3B1b9KwDgqLmz64LQsM5EJAshfBrA1+bPmY3P/tWl+PDffGGv+zIeL3vh83Hhu/6i/vL2NE1/tLvz24G2nDJccMG7H6FwM2GWZ0auV0R2aibHX3njjd17vktksvHerwLwQwC4+Oy3jx28+8wxR83Hjf/0d3X/xKj3/mMtu/lBTFsKAgA6yq8FUJG8yrGRJjATCF0Vr5jqDkbyVGze+7MArAOAL3zqw7ig5637fN9nPXMRvnHt5Zh9xOFAnov+AhGZvC2VBxHtKgjoTOVnMGQk1Tmoo1PCKQTqvD1zYODGE6a6jxFARB43s7cCGE6TBJdf9nF88TMfwbTOvQsZefNrTsd3b7kGxx97NACA5N8nSXJLC7t8UNO2grB48eJt4vlboZgZzfL6h+aQ51jURE7v7++fNdX9jAClUulXZnYqgLVAPn34jztX4YKetza9JPnCU07GN1d8GTd/6XOYOb0LAEjysiRJPj15PT/4aEunYh2rVL7vOruOdwpnEKMzMzGDiSlDmeVpb+jv77+1t7e3LYt5zOjqwmGzZu7xvK3bhsBxss+V0hTTpnXu8fox+wh2SalU+g+SL1HVQQCnzZ8zG5df9nFc9uH34js/+CnuXP1jPPTIeqx//I+oZQFzDu/GM+bNwcv/y5/izDNOqxdpqbMZwOI0Te/YY8NtRtunnv7KipWnOthLnHOK3EwwgRiKVQdzbkuX47cXL168bar7uj/IsuzVIjKh4JxjXvq6XeZDuOZzfRO5VS1Jkt0mOiUpqvoOkp8VkYnmLhgh+aUkST4nIvtnG+VBRttOGeo8/tD9P3WQ9WZWTBvEYGYsnI2iYcZo4BuXX3/9oj3fLTLZiAiTJFmVJMlzAPQAuBG737ilAFaLyF95709I0/TjUQx2TdtbCECe0NSNZO8ArGQCExMTMQPETMTyimxiYtxgWXJ3b+/Zm/Z814MTkguLBKd7Ou8lAN4KNGchkPy8c25POy7Ve/8/9qLPSa1We5Zz7igRmS8inSTXm9njaZr+PxGZUNr6diYKQsGXrr32qJKU3iBCEYHReW0IgcBM68/FhGGLiawLSbLxj/fdN5QnU20vVPW9JK8AmhME7/1xIjIzLqmOAAAHk0lEQVSh5LCR/U9bOxXHcul73rN++cqVd5rKq2EmYmZaWAcCWF65NRcF79LpTvSkUmYnHLXoeLtqxcqaJ2ueUC1Jnt1dnYmIBQnmq0YpF6+DM1VjWqJ2OmcjmtK5mon35qrOtquyy1XNzZhhYWvgqK+Y897ccGKdnTUOpan5UsmSLVusVCrZ2q4uO332bNu4cSPPOuus/Z7daH9DclqWZc92zp0gIieSPAnAQhGZSXIGgFnIf+hGSA6JyDYAG0j+3jl3n5ndlyTJmljgdXyiIIzhosWLH1l+0013WpBXwKzkSYNzuXsBMOe9CdTUNLcURAy55eBUpGROzYdE4RJDooYASyUx6VALQUzEmffBkkQsBC8jlpk4M6k5qTmISEXKZW8JZ7BSrYrr8pxemQ6PUQzPAmrbu9hlQ0zNuKm7m9XHHuP8Wo0bAfb09BjQ2urVBwIkj1TVU0meJiKnquoLnXNp8d7Y85566WH1JCsAICIgCRGBqiKE8ACAHwH4kff+RwDuPdTFtBmiIDyFi84++/H+/tu+KaUnT1Xa4Y2BL2agU4o378VEzRQw8d68qgEwRWLq1CTk14Qxz0WCMU0VITVIsDSlZaFkYGYs0Uo1mHMdVmXNKq5qruLN+4qNFNaB99stSRILc+ao27TJFgA2+5RTbPXq1dbb27vf6y5OJiTnmtnbSb5TVV8OwI0d3HXMDA+vfwxrH12PzVu2Yuv27RjaPowsC5g1cwZmTu/C9K5pWDB/Ho5feHQj50HBccXjvCKl/ENZlg2SHCyVSntfTfYgJwrCOPT2vnmE5F39K246Xix7Np13Alc4Fs0UxSA3NYFYELEkgSZKEyYWkmBQsYSJSRIsODGE1CQEE+csy8ScCyYi5jJnVfFW7qDSwzoqnVaRiqELJiPTrUuGbWgmjaVuw5YtVtq0ybq6umzNmjUGwIp0bAc9JMuq+k4A56vq6cDO6exqWYZf/+e9+Mkvf4N/v/t3uO/BtXjg4UdRrdWaur+I4Bnz5uD4hUfjuc8+CS974fPw0hc8F0d0N3LiHCsiHxWRj4YQ/kDy5iRJ+kVk/BLZhyjRqbgHBgcHS5trteNdzY6Gdw6SWwcikvsYRE20sAgaz52JhMY0IUmpzjnLMmfiMnPOFULgzPniddVbxRX+gtHEvB+xIe/Np6kdZqYdHR322PTpVl63zrq7u22q/QWtciqSnG5mHyR5KYCdEtQ8uO5RfP2Ou/DtH/wUv/rPNahUmxv8zSIiOOm4Y/GKF78Ab3vdq/GyFz4f3u+0Ep8BWOW9/xsRubeljR+gREFoksHBQf/E6Og8ZJgD72Y5M/E+UXWFKAQxKZ7XhUAKKyAEZyKZSSEENedMpGbOe+tgh1Zc1Zz35iuJeT9qw0lifntqSTJkaZrapo4OSx57zMrlsnV3d9uB4C8YKwhvXPJ+bN2+/WnnvO4VL8NnPnAxgKcLAkkxs4uLzEsNIVj/+Eas+uZ38M/f+j/49e/27xicO/sIvO21r8Lb3/AavPQFzxv7VgCwwnv/CRE5ZJecgSgIe0VfX587+uiju2pp2ulG0eGne8lqNUm1RC0pQwh03lsplBhK+fMQAsshMOvoYLY5Y3d3YrVajbWuLlYfq7Grq8o0TW10tJvDw/exu7ubXV1dtmXLFm7YsIHLli0zADxQHF9jBaEZxgoCyeNV9asAGrnLfvO7e/GP167Erd/5HoLut5owu+SUZ52AS5e8Gz1veh0S35i9bATw/iRJBqewa5NKFITIXrG3gpBl2WtF5BbkBWDxwMOP4K+/eCVu/c73xt0PMdWcdNyxuOzDl+BNrz597OGrvffvF5mE2n9TTBSEyF5BckEI4VnNnp8kyY9V9UwAXwOQkMSXVtyEz35p+aSkRms1b37N6finZR/DnCMPrx+6xXt/jogcEk7dOlEQIvuFLMteKSJ3AihvHxnBRR+7DLfdtXqquzUhFsyfi699+fN43skn1Q9dkSTJ+6eyT60mCkJk0iE5Q1V/C2BhLctw1iV/hX/78c+mult7xczpXfjWiivw/D9pGEdvTJLkW1PZp1bS9rsdI5NPCOFTABYCwAf7PnfQigEAbNs+jLdf8t/GLrNeSfKQieeJghCZVEimInIBAPzw57/CDV+/faq7tM88vvEJ/PX/vLL+cqGqvn4q+9NKoiBEJpUQwqsAzAaAq25cNcW9aR233HYHntzayJnzzqnsSyuJghCZVJxzjaxG3/+/v5jKrrSUSrWGn/767vrLiWZuOmCJghCZVMzsaAAYGh7B1m1DU92dlvLwjjL1x0xlP1pJFITIZFMCgFqTm5AOJsZ8pt3mgTyYiIIQiUQaREGIRCINoiBEIpEGURAikUiDKAiRSKTBIRNyGTmwmTVzBn70zyumuhst5ai5c6a6Cy0nCkJkv5B4P3ZDUOQAJQpCZFJxzv2e5IRqRR6EPD1/XCQSiUQikUgkEolEIpFIJBKJRCKRSCQSiUQikUgkEolEIpFIJBKJRCKRSCQSiUQikUgkEolEIpFIJBKJRCKRSCQSiUQikUgkEolEIpFIJBKJRCKRSCQSiUQikUgkEolEIpFIJBKJRA5e/j8FANjWkmDXtwAAAABJRU5ErkJggg==";
    
    private const WIDGET_TITLE = [
        self::LANGUAGE_CODE_GERMAN  => 'Neukunden',
        self::LANGUAGE_CODE_ENGLISH => 'New Customers',
    ];
    
    /**
     * @var DataFactory
     */
    private $dataFactory;
    
    /**
     * @var Connection
     */
    private $connection;
    
    /**
     * @var NumberFormatter
     */
    private $numberFormatter;
    
    
    public function __construct(
        OverviewFactory $overviewFactory,
        DataFactory     $dataFactory,
        Connection      $connection,
        NumberFormatter $numberFormatter
    ) {
        $this->dataFactory     = $dataFactory;
        $this->connection      = $connection;
        $this->numberFormatter = $numberFormatter;
        
        parent::__construct($overviewFactory->createType(self::TYPE),
                            $overviewFactory->createVersion(self::VERSION),
                            $overviewFactory->createIconUsingData(self::ICON),
                            $overviewFactory->useVisualizations()->createText(),
                            $overviewFactory->useOptions()->createOptions($overviewFactory->useOptions()
                                                                              ->usePredefined()
                                                                              ->createTimespanDropdownIncludingToday($overviewFactory)),
                            $overviewFactory->createTitles($overviewFactory->createTitle($overviewFactory->createLanguageCode(self::LANGUAGE_CODE_GERMAN),
                                                                                         self::WIDGET_TITLE[self::LANGUAGE_CODE_GERMAN]),
                                                           $overviewFactory->createTitle($overviewFactory->createLanguageCode(self::LANGUAGE_CODE_ENGLISH),
                                                                                         self::WIDGET_TITLE[self::LANGUAGE_CODE_ENGLISH])));
    }
    
    
    public function data(Options $options): Data
    {
        $timespan = $this->dataFactory->useTimespan()->createFromTerm($options->getById(TimespanDropdown::ID)->value());
        $signups  = $this->connection->createQueryBuilder()
                        ->select('count(*) AS signups')
                        ->FROM('customers')
                        ->where('account_type = 0')
                        ->andWhere('customers_date_added BETWEEN :startDate AND :endDate')
                        ->setParameters([
                                            ':startDate' => $timespan->startDate()
                                                ->format(self::QUERY_TIMESPAN_FORMAT_DATE_START),
                                            ':endDate'   => $timespan->endDate()
                                                ->format(self::QUERY_TIMESPAN_FORMAT_DATE_END),
                                        ])
                        ->execute()
                        ->fetchAll()[0];
        
        return $this->dataFactory->useTextData()->createTextData($this->dataFactory->useTextData()
                                                                     ->createValue($this->numberFormatter->format((int)($signups['signups']
                                                                                                                        ??
                                                                                                                        0))));
    }
}